<?php

namespace App\Console\Commands;

use App\Models\General_Setting;
use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class SendDailyWalletNotification extends Command
{
    protected $signature = 'notify:wallet-balance';
    protected $description = 'Send wallet balance notification to all users daily';

    public function handle()
    {
        // Fetch OneSignal settings
        $settings = General_Setting::get();
        $config = [];
        foreach ($settings as $item) {
            $config[$item->key] = $item->value;
        }

        $appId = $config['onesignal_app_id'] ?? null;
        $restKey = $config['onesignal_rest_key'] ?? null;

        if (!$appId || !$restKey) {
            return Command::FAILURE;
        }

        // Get users who have non-zero wallet and valid device info
        $users = User::where('wallet_coin', '!=', 0)->whereNotNull('device_token')->where('device_type', '!=', 0)->get();

        foreach ($users as $user) {
            $deviceToken = $user->device_token;
            $deviceType = $user->device_type;
            $balance = $user->wallet_coin;
            $userName = $user->full_name ?? $user->user_name ?? 'User';

            $message = "Hi {$userName}, your current wallet balance is ₹{$balance}";

            $fields = [
                'app_id' => $appId,
                'headings' => ['en' => config('app.name')],
                'contents' => ['en' => $message],
                'channel_for_external_user_ids' => 'push',
            ];

            switch ($deviceType) {
                case 1: // Android
                    $fields['include_android_reg_ids'] = [$deviceToken];
                    break;
                case 2: // iOS
                case 3: // Web
                    $fields['include_player_ids'] = [$deviceToken];
                    break;
                default:
                    continue 2;
            }

            $this->sendNotification($fields, $restKey);
        }

        Log::info("🏁 Wallet notification job completed.");
        return Command::SUCCESS;
    }

    private function sendNotification(array $fields, string $restKey)
    {
        $ch = curl_init();

        curl_setopt_array($ch, [
            CURLOPT_URL => "https://onesignal.com/api/v1/notifications",
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json; charset=utf-8',
                'Authorization: Basic ' . $restKey,
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($fields),
            CURLOPT_SSL_VERIFYPEER => false,
        ]);

        $response = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($status !== 200) {
            Log::error("❌ OneSignal failed. Status: {$status}. Response: {$response}");
        }
    }
}
